//
//  TodoItemEditor.swift
//  Do It
//
//  Created by Jim Dovey on 10/6/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

// START:LocalizedTitle
fileprivate let localizedNewItemTitle = NSLocalizedString(
    "New Item", comment: "default title for new to-do item")
// END:LocalizedTitle

struct TodoItemEditor: View {
    // START:NewState
    @ObservedObject var item: TodoItem
    @Environment(\.managedObjectContext) var objectContext
    @Environment(\.presentationMode) var presentationMode
    // END:NewState

    // START:NotesEditor
    var notesEditor: some View {
        TextView(text: Binding($item.notes, replacingNilWith: ""))//<label id="code.7.notes.binding.nil"/>
            .padding(.horizontal)
            .navigationBarTitle("Notes: \(item.title ?? localizedNewItemTitle)")//<label id="code.7.notes.title.nil"/>
    }
    // END:NotesEditor

    // START:DoneCancel
    var cancelButton: some View {
        Button(action: {
            self.presentationMode.wrappedValue.dismiss()//<label id="code.7.todo.editor.cancel.dismiss"/>
        }) {
            Text("Cancel")
                .foregroundColor(.accentColor)
        }
    }

    var doneButton: some View {
        Button(action: {
            try? self.objectContext.save()//<label id="code.7.todo.editor.save"/>
            self.presentationMode.wrappedValue.dismiss()//<label id="code.7.todo.editor.done.dismiss"/>
        }) {
            Text("Done")
                .bold()
                .foregroundColor(.accentColor)
        }
    }
    // END:DoneCancel

    var body: some View {
        Form {
            // START:FormUpdates
            TextField("Title", text: Binding($item.title, localizedNewItemTitle))

            Picker("List", selection: Binding(
                $item.list, TodoItemList.defaultList(in: self.objectContext))
            ) {
                ForEach(TodoItemList.allLists(in: self.objectContext)) { list in
                    Text(list.name ?? "<unknown>")
                }
            }
            // END:FormUpdates

            Picker("Priority", selection: $item.priority) {
                ForEach(Priority.allCases, id: \.self) { priority in
                    Text(priority.localizedName)
                }
            }

            HStack {
                Text("Due Date")
                Spacer()
                // START:HasDueDate
                Toggle("Has Due Date", isOn: Binding(isNotNil: $item.date, defaultValue: Date()))
                    .labelsHidden()
                // END:HasDueDate
            }

            // START:DatePicker
            // START_HIGHLIGHT
            if self.item.date != nil {
                // END_HIGHLIGHT
                Toggle("Include Time", isOn: $item.usesTimeOfDay)
                HStack {
                    Spacer()
                    // START_HIGHLIGHT
                    DatePicker("Due Date", selection: Binding($item.date, Date()),
                               // END_HIGHLIGHT
                               displayedComponents: item.usesTimeOfDay
                                   ? [.date, .hourAndMinute]
                                   : .date)
                        .datePickerStyle(WheelDatePickerStyle())
                        .labelsHidden()
                    Spacer()
                }
            }
            // END:DatePicker
            
            NavigationLink(destination: notesEditor) {
                Text("Notes")
            }
        }
        .navigationBarTitle(Text("Editing: \(item.title ?? localizedNewItemTitle)"),
                            displayMode: .inline)
        .navigationBarItems(leading: cancelButton, trailing: doneButton)
    }
}

struct TodoItemEditor_Previews: PreviewProvider {
    static var previews: some View {
        // START:Preview
        NavigationView {
            TodoItemEditor(item: PreviewDataStore.shared.sampleItem)
        }
        .environment(\.managedObjectContext, PreviewDataStore.shared.viewContext)
        // END:Preview
    }
}
